import * as state from './state.js';
import * as ipc from './ipc.js';
import * as input from './input.js';
import { t, playSound, navigateSound, confirmSound, cancelSound, backgroundMusic, anyModalOpen, fadeOutAndStop } from './utils.js';

let onProcessStartedCallback = () => {};
let scrollAnimationId = null;

export function init(callbacks) {
    onProcessStartedCallback = callbacks.onProcessStarted;
}

export const gameGrid = document.getElementById('game-grid');
export const gameTitleElement = document.getElementById('game-title');
export const playerCountElement = document.getElementById('player-count');
export const saveStateQuadrants = document.querySelectorAll('.savestate-quadrant');
export const mainLogo = document.getElementById('main-logo');

export const actionIconGear = document.getElementById('action-icon-gear');
export const actionIconWifi = document.getElementById('action-icon-wifi');
export const actionIconConsole = document.getElementById('action-icon-console');

const actionIcons = [actionIconConsole, actionIconWifi, actionIconGear];

export const startGameModal = document.getElementById('start-menu-modal');
export const systemOptionsModal = document.getElementById('system-options-modal');
export const ingameOptionsModal = document.getElementById('ingame-options-modal');
export const generalOptionsModal = document.getElementById('general-options-modal');
export const networkFeaturesModal = document.getElementById('network-features-modal');
export const lobbyModal = document.getElementById('lobby-modal');
export const powerMenuModal = document.getElementById('power-menu-modal');
export const saveStatesModal = document.getElementById('save-states-modal');
export const netplayUsernameModal = document.getElementById('netplay-username-modal');
export const raLoginModal = document.getElementById('ra-login-modal');
export const consoleMenuModal = document.getElementById('console-menu-modal');
export const progressOverlay = document.getElementById('progress-overlay');
export const attractModeOverlay = document.getElementById('attract-mode-overlay');
export const attractSonic = document.getElementById('attract-sonic');

export const startMenuOptions = startGameModal.querySelectorAll('.menu-button');
export const systemOptionsList = systemOptionsModal.querySelectorAll('.menu-button');
export const ingameOptionsList = ingameOptionsModal.querySelectorAll('.option-item, .menu-button');
export const generalOptionsList = generalOptionsModal.querySelectorAll('.option-item');
export const networkFeaturesList = networkFeaturesModal.querySelectorAll('.option-item, .menu-button');
export const lobbyList = document.getElementById('lobby-list');
export const powerMenuList = powerMenuModal.querySelectorAll('.menu-button');
export const consoleMenuList = consoleMenuModal.querySelectorAll('.menu-button');
export const saveStatesSlotContainers = saveStatesModal.querySelectorAll('.save-slot-container');
export const saveStatesSlots = saveStatesModal.querySelectorAll('.save-slot');
export const saveStatesTimestamps = saveStatesModal.querySelectorAll('.save-slot-timestamp');
export const saveStatesActions = saveStatesModal.querySelectorAll('.action-buttons .menu-button');
export const netplayUsernameActions = netplayUsernameModal.querySelectorAll('.action-buttons .menu-button');
export const raLoginActions = raLoginModal.querySelectorAll('.action-buttons .menu-button');

export const bgmVolumeBarFill = document.querySelector('.volume-bar-fill');
export const bgmNameDisplay = document.getElementById('bgm-name-display');
export const startFullscreenDisplay = document.getElementById('start-fullscreen-display');
export const languageDisplay = document.getElementById('language-display');
export const regionDisplay = document.getElementById('region-display');
export const attractModeDisplay = document.getElementById('attract-mode-display');
export const saveSystemToggleDisplay = document.getElementById('save-system-toggle-display');
export const aspectRatioDisplay = document.getElementById('aspect-ratio-display');
export const videoFullscreenDisplay = document.getElementById('video-fullscreen-display');
export const scanlinesDisplay = document.getElementById('scanlines-display');
export const scanlinesStrengthDisplay = document.getElementById('scanlines-strength-display');
export const rewindFfDisplay = document.getElementById('rewind-ff-display');
export const raToggleDisplay = document.getElementById('ra-toggle-display');
export const raLoginButton = document.getElementById('ra-login-button');
export const relayServerDisplay = document.getElementById('relay-server-display');
export const usernameInput = document.getElementById('username-input');
export const raUsernameInput = document.getElementById('ra-username-input');
export const raPasswordInput = document.getElementById('ra-password-input');
export const progressStatus = document.getElementById('progress-status');
export const progressBar = document.getElementById('progress-bar');
export const progressPercent = document.getElementById('progress-percent');

export function renderUI() {
    gameGrid.innerHTML = '';
    const gridFragment = document.createDocumentFragment();
    state.games.forEach((game, index) => {
        const coverItem = document.createElement('div');
        coverItem.className = 'game-cover';
        if (game.coverUrl) {
            coverItem.style.backgroundImage = `url("${game.coverUrl}")`;
        }
        coverItem.addEventListener('click', () => {
            updateSelection(index);
            openStartGameMenu();
        });
        gridFragment.appendChild(coverItem);
    });
    gameGrid.appendChild(gridFragment);
}

function removeSelectionCorners() {
    const existingCorners = document.querySelectorAll('.selection-corner');
    existingCorners.forEach(corner => corner.remove());
}

function addSelectionCorners(element) {
    removeSelectionCorners();
    if (!element) return;
    const tl = document.createElement('div'); tl.className = 'selection-corner tl';
    const tr = document.createElement('div'); tr.className = 'selection-corner tr';
    const bl = document.createElement('div'); bl.className = 'selection-corner bl';
    const br = document.createElement('div'); br.className = 'selection-corner br';
    element.append(tl, tr, bl, br);
}

function smoothScrollToElement(container, element, instant = false) {
    if (!element) return;

    if (scrollAnimationId) {
        cancelAnimationFrame(scrollAnimationId);
    }

    const containerRect = container.getBoundingClientRect();
    const elementRect = element.getBoundingClientRect();
    const startPosition = container.scrollTop;
    let endPosition = startPosition;

    if (elementRect.top < containerRect.top) {
        endPosition = startPosition + (elementRect.top - containerRect.top);
    } else if (elementRect.bottom > containerRect.bottom) {
        endPosition = startPosition + (elementRect.bottom - containerRect.bottom);
    } else {
        if (!instant) return;
    }

    if (instant) {
        container.scrollTop = endPosition;
        return;
    }

    const distance = endPosition - startPosition;
    const duration = 120;
    let startTime = null;

    function animationStep(currentTime) {
        if (startTime === null) startTime = currentTime;
        const timeElapsed = currentTime - startTime;
        const progress = Math.min(timeElapsed / duration, 1);
        const ease = 1 - Math.pow(1 - progress, 3);
        container.scrollTop = startPosition + distance * ease;
        if (timeElapsed < duration) {
            scrollAnimationId = requestAnimationFrame(animationStep);
        }
    }
    scrollAnimationId = requestAnimationFrame(animationStep);
}

export async function updateSaveStateIcon(gameIndex) {
    if (gameIndex < 0 || gameIndex >= state.games.length) return;
    const gameBaseName = state.games[gameIndex].zipName.replace(/\.(zip|chd)$/i, '');
    
    const updateIcon = (states) => {
        saveStateQuadrants.forEach((quadrant, i) => {
            quadrant.classList.toggle('filled', states[i] !== null);
        });
    };

    if (state.saveStateCache[gameBaseName] !== undefined && state.saveStateCache[gameBaseName] !== 'fetching') {
        updateIcon(state.saveStateCache[gameBaseName]);
        return;
    }

    if (state.saveStateCache[gameBaseName] === 'fetching') return;

    state.saveStateCache[gameBaseName] = 'fetching';
    const fetchedStates = await ipc.manageSaveState({ action: 'scan', gameBaseName });
    state.saveStateCache[gameBaseName] = fetchedStates;
    updateIcon(fetchedStates);
}

export function updateActionIconSelection() {
    if (state.focusedElement === 'actions') {
        const oldSelectedItem = gameGrid.children[state.selectedIndex];
        if (oldSelectedItem) oldSelectedItem.classList.remove('selected');
        removeSelectionCorners();
    }
    actionIcons.forEach((icon, index) => {
        const isActive = state.focusedElement === 'actions' && index === state.actionIconIndex;
        icon.classList.toggle('active', isActive);
    });
}


export function updateSelection(newIndex, playSoundEffect = true, instantScroll = false) {
    state.setFocusedElement('grid');
    updateActionIconSelection();

    if (state.games.length === 0) {
        gameTitleElement.textContent = t('game_title_no_games');
        playerCountElement.textContent = '0';
        removeSelectionCorners();
        saveStateQuadrants.forEach(q => q.classList.remove('filled'));
        return;
    };

    if (playSoundEffect && state.selectedIndex !== newIndex) {
        playSound(navigateSound);
    }

    const oldSelectedItem = gameGrid.children[state.selectedIndex];
    if (oldSelectedItem) oldSelectedItem.classList.remove('selected');

    state.setSelectedIndex(newIndex);

    const newSelectedItem = gameGrid.children[state.selectedIndex];
    if (!newSelectedItem) return;

    newSelectedItem.classList.add('selected');
    addSelectionCorners(newSelectedItem);
    
    smoothScrollToElement(gameGrid, newSelectedItem, instantScroll);

    const selectedGame = state.games[state.selectedIndex];
    gameTitleElement.textContent = selectedGame.baseName;
    const isTwoPlayer = state.twoPlayerGames.includes(selectedGame.baseName);
    playerCountElement.textContent = isTwoPlayer ? '2' : '1';
    
    updateSaveStateIcon(state.selectedIndex);
    
    state.appSettings.lastGameIndex = state.selectedIndex;
}

function updateActiveItem(elements, index) {
    elements.forEach((item, i) => {
        item.classList.toggle('active', i === index);
    });
}

export function updateMainLogo() {
    const region = state.appSettings.logoRegion || 'America';
    const logoFile = state.currentPlatform.logos[region] || state.currentPlatform.logos['America'];
    const logoUrl = `../../resources/images/ui/${logoFile}`;
    
    
    mainLogo.style.opacity = '0';
    setTimeout(() => {
        mainLogo.src = logoUrl;
        mainLogo.style.opacity = '1';
    }, 200);
}

export async function switchPlatform(platformKey) {
    const platform = state.PLATFORMS[platformKey];
    if (!platform || state.currentPlatform.id === platform.id) return;
    
    state.setCurrentPlatform(platform);
    
    
    state.appSettings.lastPlatformId = platform.id;

    updateMainLogo();

    
    const newData = await ipc.getInitialData(platform.id);
    if (!newData) return;
    
    state.setInitialData(newData);
    
    
    renderUI();
    
    updateSelection(0, false, true);
    
    if (state.appSettings.bgmPath === "off") {
        
    } else {
        const trackName = state.availableBgmTracks[state.appSettings.currentBgmIndex]?.name;
        let newIndex = state.availableBgmTracks.findIndex(t => t.name === trackName);
        if (newIndex === -1) newIndex = 1; 
        state.appSettings.currentBgmIndex = newIndex;
    }
}

export function openStartGameMenu() {
    if (state.focusedElement !== 'grid' || anyModalOpen() || state.games.length === 0) return;
    state.setStartMenuOpen(true);
    playSound(confirmSound);
    startGameModal.classList.remove('hidden');

    const saveStatesButton = document.getElementById('open-savestates-menu-button');
    const isSaveSystemEnabled = state.appSettings.savestate_auto_index === 'true';
    saveStatesButton.style.display = isSaveSystemEnabled ? 'block' : 'none';

    
    const netplayButton = document.getElementById('start-game-netplay-button');
    const isNetplaySupported = state.currentPlatform.id === 'Sega Genesis';
    netplayButton.style.display = isNetplaySupported ? 'block' : 'none';

    state.setStartMenuSelectionIndex(0);
    updateActiveItem(startMenuOptions, 0);
}

export function closeStartGameMenu(playSoundEffect = true) {
    if (!state.isStartMenuOpen) return;
    state.setStartMenuOpen(false);
    if(playSoundEffect) playSound(cancelSound);
    startGameModal.classList.add('hidden');
    input.resetInactivityTimer();
}

export function openSystemOptionsMenu(resetPosition = true) {
    if (anyModalOpen()) return;
    state.setSystemOptionsMenuOpen(true);
    playSound(confirmSound);
    systemOptionsModal.classList.remove('hidden');
    if (resetPosition) {
        state.setSystemOptionsSelectionIndex(0);
    }
    updateActiveItem(systemOptionsList, state.systemOptionsSelectionIndex);
}

export function closeSystemOptionsMenu(playSoundEffect = true) {
    if (!state.isSystemOptionsMenuOpen) return;
    state.setSystemOptionsMenuOpen(false);
    if(playSoundEffect) playSound(cancelSound);
    systemOptionsModal.classList.add('hidden');
    input.resetInactivityTimer();
}

export function openIngameOptionsMenu() { closeSystemOptionsMenu(false); state.setIngameOptionsOpen(true); populateIngameOptionsMenu(); ingameOptionsModal.classList.remove('hidden'); state.setIngameOptionsSelectionIndex(0); updateActiveItem(ingameOptionsList, 0); }

export function closeIngameOptionsMenu() { state.setIngameOptionsOpen(false); ingameOptionsModal.classList.add('hidden'); ipc.saveAllSettings(state.appSettings); openSystemOptionsMenu(false); }
export function openGeneralOptionsMenu() { 
    closeSystemOptionsMenu(false); 
    state.setGeneralOptionsOpen(true); 
    populateGeneralMenu(); 
    generalOptionsModal.classList.remove('hidden'); 
    state.setGeneralOptionsSelectionIndex(0); 
    updateActiveItem(generalOptionsList, 0); 
}

export function closeGeneralOptionsMenu() { state.setGeneralOptionsOpen(false); generalOptionsModal.classList.add('hidden'); ipc.saveAppSettings(state.appSettings); openSystemOptionsMenu(false); }
export async function openNetworkFeaturesMenu() {
    closeSystemOptionsMenu(false);
    state.setNetworkFeaturesOpen(true);
    networkFeaturesModal.classList.remove('hidden');
    state.setNetworkFeaturesSelectionIndex(0);
    updateActiveItem(networkFeaturesList, 0);
    await populateNetworkMenu();
}

export function closeNetworkFeaturesMenu(playSoundEffect = true) {
    if (!state.isNetworkFeaturesOpen) return;
    if (state.currentRelayIndex !== state.initialRelayIndex) { const newConfig = state.relayOptions[state.currentRelayIndex]; ipc.setRelayConfig(newConfig); } 
    state.setNetworkFeaturesOpen(false); 
    networkFeaturesModal.classList.add('hidden'); 
    ipc.saveAllSettings(state.appSettings); 
    if (playSoundEffect) openSystemOptionsMenu(false); 
}

export async function openLobbyMenu(origin) {
    if (origin === 'networkMenu') {
        closeNetworkFeaturesMenu(false);
    } else if (origin === 'startMenu') {
        closeStartGameMenu(false);
    } else if (origin === 'systemMenu') {
        closeSystemOptionsMenu(false);
    }
    state.setLobbyMenuOrigin(origin);
    state.setLobbyOpen(true);
    playSound(confirmSound);
    lobbyList.innerHTML = `<div class="lobby-item">${t('lobbymenu_fetching')}</div>`;
    lobbyModal.classList.remove('hidden');
    try {
        const hostedGames = await ipc.getLobbyList();
        renderLobbyList(hostedGames);
    } catch (error) {
        lobbyList.innerHTML = `<div class="lobby-item">${t('lobbymenu_error')}</div>`;
    }
}

export function closeLobbyMenu() {
    state.setLobbyOpen(false);
    lobbyModal.classList.add('hidden');
    if (state.lobbyMenuOrigin === 'networkMenu') {
        openNetworkFeaturesMenu();
    } else if (state.lobbyMenuOrigin === 'startMenu') {
        openStartGameMenu();
    } else if (state.lobbyMenuOrigin === 'systemMenu') {
        openSystemOptionsMenu();
    }
    state.setLobbyMenuOrigin(null);
}

export function openPowerMenu() { closeSystemOptionsMenu(false); state.setPowerMenuOpen(true); powerMenuModal.classList.remove('hidden'); state.setPowerMenuSelectionIndex(0); updateActiveItem(powerMenuList, 0); }

export function closePowerMenu(reopenParent = true) { 
    state.setPowerMenuOpen(false); 
    powerMenuModal.classList.add('hidden'); 
    if (reopenParent) openSystemOptionsMenu(false); 
}

export async function openSaveStatesMenu() { closeStartGameMenu(false); state.setSaveStatesMenuOpen(true); saveStatesModal.classList.remove('hidden'); await populateSaveStatesMenu(); state.setSaveStatesNavRow(0); state.setSaveStatesSlotSelectionIndex(0); updateSaveStatesSelection(); }

export function closeSaveStatesMenu(playSoundEffect = true) { if (!state.isSaveStatesMenuOpen) return; state.setSaveStatesMenuOpen(false); if (playSoundEffect) playSound(cancelSound); saveStatesModal.classList.add('hidden'); openStartGameMenu(); }

export function openNetplayUsernameModal() {
    closeNetworkFeaturesMenu(false);
    state.setNetplayUsernameOpen(true);
    netplayUsernameModal.classList.remove('hidden');
    usernameInput.value = state.appSettings.netplayUsername || "Player1";
    state.setNetplayUsernameSelectionIndex(1);
    updateActiveItem(netplayUsernameActions, 0);
}

export function closeNetplayUsernameModal() {
    state.setNetplayUsernameOpen(false);
    netplayUsernameModal.classList.add('hidden');
    openNetworkFeaturesMenu();
}

export function openRaLoginModal() {
    closeNetworkFeaturesMenu(false);
    state.setRaLoginOpen(true);
    raLoginModal.classList.remove('hidden');
    raUsernameInput.value = state.appSettings.cheevos_username;
    raPasswordInput.value = state.appSettings.cheevos_password;
    state.setRaLoginSelectionIndex(2);
    updateActiveItem(raLoginActions, 0);
}

export function closeRaLoginModal() {
    state.setRaLoginOpen(false);
    raLoginModal.classList.add('hidden');
    openNetworkFeaturesMenu();
}

export function openConsoleMenu() {
    state.setConsoleMenuOpen(true);
    playSound(confirmSound);
    consoleMenuModal.classList.remove('hidden');
    state.setConsoleMenuSelectionIndex(0);
    updateActiveItem(consoleMenuList, 0);
}

export function closeConsoleMenu(playSoundEffect = true) {
    if (!state.isConsoleMenuOpen) return;
    state.setConsoleMenuOpen(false);
    if(playSoundEffect) playSound(cancelSound);
    consoleMenuModal.classList.add('hidden');
    input.resetInactivityTimer();
}

export function populateIngameOptionsMenu() {
    saveSystemToggleDisplay.textContent = state.appSettings.savestate_auto_index === 'true' ? t('common_enable') : t('common_disable');
    aspectRatioDisplay.textContent = state.aspectLevels[state.appSettings.aspect_ratio_index] || '4:3';
    videoFullscreenDisplay.textContent = state.appSettings.video_fullscreen === 'true' ? t('common_yes') : t('common_no');
    scanlinesDisplay.textContent = state.appSettings.input_overlay_enable === 'true' ? t('common_on') : t('common_off');
    const strengthKey = `configmenu_strength_${state.strengthLevels[state.appSettings.input_overlay_opacity]?.toLowerCase() || 'normal'}`;
    scanlinesStrengthDisplay.textContent = t(strengthKey);
    rewindFfDisplay.textContent = state.appSettings.rewind_enable === 'true' ? t('common_yes') : t('common_no');
}

export function populateGeneralMenu() {
    startFullscreenDisplay.textContent = state.appSettings.startFullscreen ? t('displaymenu_fullscreen') : t('displaymenu_windowed');
    const currentLang = state.availableLanguages.find(l => l.code === state.appSettings.language);
    languageDisplay.textContent = currentLang ? currentLang.name : 'English';
    
   
    regionDisplay.textContent = state.appSettings.logoRegion;
    
    attractModeDisplay.textContent = state.appSettings.attractModeEnabled === 'true' ? t('common_on') : t('common_off');

    bgmVolumeBarFill.style.width = `${state.appSettings.bgmVolumeLevel * 10}%`;
    const track = state.availableBgmTracks[state.appSettings.currentBgmIndex];
    if (track) {
        if (track.name === 'Music Off') bgmNameDisplay.textContent = t('audiomenu_bgm_off');
        else if (track.name === 'Default') bgmNameDisplay.textContent = t('audiomenu_bgm_default');
        else bgmNameDisplay.textContent = track.name;
    }
    
    const regionLabel = document.querySelector('[data-i18n-key="displaymenu_select_region"]');
    if (regionLabel) {
        regionLabel.textContent = "Logo Region";
    }
}

export async function populateNetworkMenu() {
    raToggleDisplay.textContent = state.appSettings.cheevos_enable === 'true' ? t('common_enable') : t('common_disable');
    raLoginButton.classList.toggle('disabled', state.appSettings.cheevos_enable !== 'true');
    const currentConfig = await ipc.getRelayConfig();
    const foundIndex = state.relayOptions.findIndex(opt => opt.server === currentConfig.server && opt.enabled === currentConfig.enabled);
    const newIndex = (foundIndex !== -1) ? foundIndex : 0;
    state.setCurrentRelayIndex(newIndex);
    state.setInitialRelayIndex(newIndex);
    updateRelayServerDisplay();
}

export async function populateSaveStatesMenu() {
    if (state.games.length === 0) return;
    const gameBaseName = state.games[state.selectedIndex].zipName.replace(/\.(zip|chd)$/i, '');
    const fetchedStates = await ipc.manageSaveState({ action: 'scan', gameBaseName });
    state.setCurrentSaveStates(fetchedStates);

    for (let i = 0; i < saveStatesSlots.length; i++) {
        const slot = saveStatesSlots[i];
        const timestamp = saveStatesTimestamps[i];
        const stateData = state.currentSaveStates[i];
        
        slot.style.backgroundImage = '';
        slot.textContent = '';
        timestamp.textContent = '';

        if (stateData) {
            const fileSlot = stateData.fileSlot;
            const slotSuffix = fileSlot > 0 ? `${fileSlot}` : '';
            const pngPath = `${state.APP_PATHS.statesDir}/${gameBaseName}.state${slotSuffix}.png`.replace(/\\/g, '/');
            const cacheBuster = `?t=${Date.now()}`;
            slot.style.backgroundImage = `url('file:///${pngPath}${cacheBuster}')`;
             if (stateData.mtime) {
                const date = new Date(stateData.mtime);
                const dateString = date.toLocaleDateString(undefined, { year: '2-digit', month: '2-digit', day: '2-digit' });
                const timeString = date.toLocaleTimeString(undefined, { hour: 'numeric', minute: '2-digit', hour12: true });
                timestamp.textContent = `${dateString}, ${timeString}`;
            }
        } else {
            slot.textContent = 'No Data';
        }
    }
}

export function updateSaveStatesSelection() {
    saveStatesSlots.forEach((slot, i) => {
        slot.classList.toggle('active', state.saveStatesNavRow === 0 && i === state.saveStatesSlotSelectionIndex);
    });
    saveStatesActions.forEach((button, i) => {
        button.classList.toggle('active', state.saveStatesNavRow === 1 && i === state.saveStatesActionSelectionIndex);
    });
}

export function updateRelayServerDisplay() {
    const currentOption = state.relayOptions[state.currentRelayIndex];
    relayServerDisplay.textContent = t(currentOption.labelKey);
}

export function renderLobbyList(hostedGames) {
    lobbyList.innerHTML = '';
    state.setLobbyGames(hostedGames);
    if (state.lobbyGames.length === 0) {
        lobbyList.innerHTML = `<div class="lobby-item">${t('lobbymenu_no_games')}</div>`;
        return;
    }
    const fragment = document.createDocumentFragment();
    state.lobbyGames.forEach(game => {
        const item = document.createElement('div');
        item.className = 'lobby-item';
        item.innerHTML = `<span class="game-name">${game.game_name}</span><span class="host-name">${t('lobbymenu_hosted_by', { host: game.username })}</span>`;
        fragment.appendChild(item);
    });
    lobbyList.appendChild(fragment);
    state.setLobbySelectionIndex(0);
    updateLobbySelection();
}

export function updateLobbySelection() {
    Array.from(lobbyList.children).forEach((item, index) => {
        const isSelected = index === state.lobbySelectionIndex;
        item.classList.toggle('active', isSelected);
        if (isSelected) {
            item.scrollIntoView({ block: 'nearest' });
        }
    });
}

export function changeBgm(direction) {
    if (state.availableBgmTracks.length === 0) return;
    state.appSettings.currentBgmIndex = (state.appSettings.currentBgmIndex + direction + state.availableBgmTracks.length) % state.availableBgmTracks.length;
    const track = state.availableBgmTracks[state.appSettings.currentBgmIndex];
    if (track.name === 'Music Off') {
        state.appSettings.bgmPath = "off";
        backgroundMusic.pause();
    } else {
        state.appSettings.bgmPath = track.name === 'Default' ? "default" : track.path;
        if (track.path) {
            backgroundMusic.src = `file:///${track.path.replace(/\\/g, '/')}`;
            backgroundMusic.play().catch(e => console.error("BGM failed to play:", e));
        }
    }
    populateGeneralMenu();
    playSound(navigateSound);
}

export function showProgressOverlay(initialMessage) {
    state.setProcessActive(true);
    onProcessStartedCallback();
    fadeOutAndStop(backgroundMusic);
    
    if (state.isSaveStatesMenuOpen) closeSaveStatesMenu(false);
    if (state.isStartMenuOpen) closeStartGameMenu(false);
    if (state.isSystemOptionsMenuOpen) closeSystemOptionsMenu(false);
    if (state.isNetworkFeaturesOpen) closeNetworkFeaturesMenu(false);
    if (state.isConsoleMenuOpen) closeConsoleMenu(false);
    
    progressOverlay.classList.remove('hidden');
    progressStatus.textContent = initialMessage;
    progressBar.style.width = '0%';
    progressPercent.textContent = '';
}

export function hideProgressOverlay() {
    progressOverlay.classList.add('hidden');
}